import boto3
import json
import os
import requests
from datetime import datetime, date, time
from elasticsearch import Elasticsearch
from elasticsearch.client import ClusterClient
from botocore.exceptions import ClientError


def lambda_handler(event, context):
  OUTBOUND_TOPIC_ARN = os.environ['outbound_topic_arn']
  GTE = os.environ['es_query_range_value']

  es = Elasticsearch(os.environ['es_domain'])
  
  violation = {}
  violation["loginfailures"] = elevated_login_failures(es, event, GTE)
  violation["rootuserapi"] = root_user_api(es, event, GTE)
  violation["sensitiverds"] = sensitive_rds_calls(es, event, GTE)
  violation["alteredlogsources"] = altered_log_sources(es, event, GTE)
  violation["s3loggingaccess"] = s3_logging_bucket_access(es, event, GTE)
  violation["vpcresources"] = vpc_resource_create_delete(es, event, GTE)
  violation["sgnaclresources"] = sg_nacl_create_delete(es, event, GTE)
  violation["iamresources"] = iam_create_delete(es, event, GTE)
  violation["routetablesresources"] = route_table_create_delete(es, event, GTE)
  violation["subnetresources"] = subnet_create_delete(es, event, GTE)

  messagebody = construct_violation_message(violation, GTE)
  print(messagebody)
  
  send_violation(messagebody, event, context, OUTBOUND_TOPIC_ARN)

def elevated_login_failures(es, event, GTE):
  message = {}
  message["title"] = "All Failed login attempts"
  api_search_query = es.search(index='cwl-*', body={
    "query": {
      "bool": {
        "must": [
          {
            "query_string": {
              "default_field": "*",
              "query": "responseElements.ConsoleLogin:Failure"
            }
          },
          {
            "range": {
              "@timestamp": {
                "gte": GTE
              }
            }
          }
        ]
      }
    }
  })
  
  api_search_query = json.dumps(api_search_query)
  api_search_query = json.loads(api_search_query)
  api_search_query = api_search_query['hits']['total']
  
  if api_search_query > 0:
    message["body"] = str(api_search_query) + ' Failed login attempts'
  return message

def root_user_api(es, event, GTE):
  message = {}
  message["title"] = "All Commands, API action taken by AWS root user"
  api_search_query = es.search(index='cwl-*', body={
    "query": {
      "bool": {
        "must": [
          {
            "query_string": {
              "default_field": "*",
              "query": "userIdentity.type:Root"
            }
          },
          {
            "range": {
              "@timestamp": {
                "gte": GTE
              }
            }
          }
        ]
      }
    }
  })
  
  api_search_query = json.dumps(api_search_query)
  api_search_query = json.loads(api_search_query)
  api_search_query = api_search_query['hits']['total']
  
  if api_search_query > 0:
    message["body"] = str(api_search_query) + ' API calls have been made by the Root user'
  return message
    
def sensitive_rds_calls(es, event, GTE):
  message = {}
  message["title"] = "Action related to RDS (configuration changes)"
  api_search_query = es.search(index='cwl-*', body={
  "query": {
    "bool": {
      "must": [
        {
          "match_all": {}
        },
        {
          "bool": {
            "must": {
              "query_string": {
                "default_field": "eventSource",
                "query": "rds.amazonaws.com"
              }
            },
            "must_not": {
              "query_string": {
                "default_field": "eventName",
                "query": "Describe* OR List*"
              }
            }
          }
        },
          {
            "range": {
              "@timestamp": {
                "gte": GTE
              }
            }
          }
        ]
      }
    }
  })
  
  api_search_query = json.dumps(api_search_query)
  api_search_query = json.loads(api_search_query)
  api_search_query = api_search_query['hits']['total']
  
  if api_search_query > 0:
    message["body"] = str(api_search_query) + ' potentially alarming RDS API call(s) in the last 24 hours'
  return message
  
def altered_log_sources(es, event, GTE):
  message = {}
  message["title"] = "Action related to enabling/disabling/changing of CloudTrail CloudWatch logs"
  api_search_query = es.search(index='cwl-*', body={
    "query": {
        "bool": {
            "must": [
                {
                    "match_all": {}
                },
                {
                    "bool": {
                        "must": {
                            "query_string": {
                                "default_field": "eventSource",
                                "query": "logs.amazonaws.com or cloudtrail.amazonaws.com"
                            }
                        },
                        "must_not": {
                            "query_string": {
                                "default_field": "eventName",
                                "query": "Describe* or Get* or List* or Test* or CreateLogStream or LookupEvents"
                            }
                        }
                    }
                },
                {
                    "range": {
                        "@timestamp": {
                            "gte": GTE
                        }
                    }
                }
            ]
        }
    }
  })
  
  api_search_query = json.dumps(api_search_query)
  api_search_query = json.loads(api_search_query)
  api_search_query = api_search_query['hits']['total']
  
  if api_search_query > 0:
    message["body"] = str(api_search_query) + ' API calls indicating alteration of log sources detected'
  return message
  
def s3_logging_bucket_access(es, event, GTE):
  message = {}
  message["title"] = "All access to S3 bucket that stores the AWS logs"
  api_search_query = es.search(index='cwl-*', body={
      "query": {
        "bool": {
          "must": [
            {
              "query_string": {
                "query": "put* OR delete*",
                "analyze_wildcard": "true",
                "default_field": "*"
              }
            },
            {
              "match_phrase": {
                "eventSource.keyword": {
                  "query": "s3.amazonaws.com"
                }
              }
            },
            {
              "match_phrase": {
                "requestParameters.bucketName.keyword": {
                  "query": "security-aod-cloudtrail"
                }
              }
            },
            {
              "range": {
                "@timestamp": {
                  "gte": GTE
                }
              }
            }
          ],
          "filter": [],
          "should": [],
          "must_not": [
            {
              "match_phrase": {
                "userIdentity.invokedBy.keyword": {
                  "query": "cloudtrail.amazonaws.com"
                }
              }
            },
            {
              "match_phrase": {
                "userAgent": {
                  "query": "cloudtrail.amazonaws.com"
                }
              }
            }
          ]
        }
      }})
  
  api_search_query = json.dumps(api_search_query)
  api_search_query = json.loads(api_search_query)
  api_search_query = api_search_query['hits']['total']
  
  if api_search_query > 0:
    message["body"] = str(api_search_query) + ' API calls indicating alteration of S3 log buckets detected'
  return message
  
def vpc_resource_create_delete(es, event, GTE):
  message = {}
  message["title"] = "Action related to VPCs (creation, deletion and changes)"
  api_search_query = es.search(index='cwl-*', body={
    "query": {
        "bool": {
          "must": [
            {
              "match_all": {}
            },
            {
              "bool": {
                "must": {
                  "query_string": {
                    "default_field": "eventName",
                    "query": "*Vpc* or *Gateway*"
                  }
                },
                "must_not": {
                  "query_string": {
                    "default_field": "eventName",
                    "query": "Describe* or *Customer*"
                  }
                }
              }
            },
            {
              "range": {
                "@timestamp": {
                  "gte": GTE
                }
              }
            }
          ]
        }
      }})
  
  api_search_query = json.dumps(api_search_query)
  api_search_query = json.loads(api_search_query)
  api_search_query = api_search_query['hits']['total']
  
  if api_search_query > 0:
    message["body"] = str(api_search_query) + ' API calls indicating creation, alteration or deletion of VPC resources'
  return message
  
def sg_nacl_create_delete(es, event, GTE):
  message = {}
  message["title"] = "Action related to changes to SGs/NACLs (creation, deletion and changes)"
  api_search_query = es.search(index='cwl-*', body={
    "query": {
        "bool": {
          "must": [
            {
              "match_all": {}
            },
            {
              "bool": {
                "must": {
                  "query_string": {
                    "default_field": "eventName",
                    "query": "*Acl* or *SecurityGroup*"
                  }
                },
                "must_not": {
                  "query_string": {
                    "default_field": "eventName",
                    "query": "Describe* or Get* or List*"
                  }
                }
              }
            },
            {
              "range": {
                "@timestamp": {
                  "gte": GTE
                }
              }
            }
          ]
        }
      }})
  
  api_search_query = json.dumps(api_search_query)
  api_search_query = json.loads(api_search_query)
  api_search_query = api_search_query['hits']['total']
  
  if api_search_query > 0:
    message["body"] = str(api_search_query) + ' API calls indicating creation, alteration or deletion of Security Group or NACL resources'
  return message

def iam_create_delete(es, event, GTE):
  message = {}
  message["title"] = "Action related to changes to IAM roles, users, and groups (creation, deletion and changes)"
  api_search_query = es.search(index='cwl-*', body={
    "query": {
      "bool": {
        "must": [{
          "query_string": {
            "default_field": "eventSource",
            "query": "iam.amazonaws.com"
          }
        },
        {
          "range": {
            "@timestamp": {
            "gte": GTE
            }
          }
        }],
        "must_not": {
          "query_string": {
            "default_field": "eventName",
            "query": "Describe* or Get* or List*"
          }
        }
      }
    }
  })
  
  api_search_query = json.dumps(api_search_query)
  api_search_query = json.loads(api_search_query)
  api_search_query = api_search_query['hits']['total']
  
  if api_search_query > 0:
    message["body"] = str(api_search_query) + ' API calls indicating creation, alteration or deletion of IAM roles, users, and groups'
  return message

def route_table_create_delete(es, event, GTE):
  message = {}
  message["title"] = "Action related to changes to Route Tables (creation, deletion and changes)"
  api_search_query = es.search(index='cwl-*', body={
    "query": {
      "bool": {
        "must": [{
          "query_string": {
            "default_field": "eventName",
            "query": "*Route*"
          }
        },
        {
          "range": {
            "@timestamp": {
            "gte": GTE
            }
          }
        }],
        "must_not": {
          "query_string": {
            "default_field": "eventName",
            "query": "Describe*"
          }
        }
      }
    }
  })

  api_search_query = json.dumps(api_search_query)
  api_search_query = json.loads(api_search_query)
  api_search_query = api_search_query['hits']['total']
  
  if api_search_query > 0:
    message["body"] = str(api_search_query) + ' API calls indicating creation, alteration or deletion of Route Tables'
  return message

def subnet_create_delete(es, event, GTE):
  message = {}
  message["title"] = "Action related to changes to Subnets (creation, deletion and changes)"
  api_search_query = es.search(index='cwl-*', body={
    "query": {
      "bool": {
        "must": [{
          "query_string": {
            "default_field": "eventName",
            "query": "*Subnet*"
          }
        },
        {
          "range": {
            "@timestamp": {
            "gte": GTE
            }
          }
        }],
        "must_not": {
          "query_string": {
            "default_field": "eventName",
            "query": "Describe*"
          }
        }
      }
    }
  })

  api_search_query = json.dumps(api_search_query)
  api_search_query = json.loads(api_search_query)
  api_search_query = api_search_query['hits']['total']
  
  if api_search_query > 0:
    message["body"] = str(api_search_query) + ' API calls indicating creation, alteration or deletion of Subnets'
  return message

def construct_violation_message(violation, GTE):
  dt = datetime.now()
  body =  "Logging Compliance Report - " + dt.strftime("%A, %d. %B %Y %I:%M%p") + "\n"
  body += "Violations for time period: '" + GTE + "'\n\n"
  for check in violation:
    if "title" in violation[check]:
      body += violation[check]["title"] + "\n"
    if "body" in violation[check]:
      if isinstance(violation[check]["body"], list) is True:
        if violation[check] == []:
          pass
        else:
          for line in violation[check]["body"]:
            body += " - " + line + "\n"
      else:
        body += " - " + violation[check]["body"] + "\n"
    else:
      body += " - No Alerts Found\n"    
  return body
  
def send_violation(messagebody, event, context, OUTBOUND_TOPIC_ARN):
    """
    Send Violation.

    Appends additional information to the message from the Lambda Context
    Sends the message created using an API call to Amazon SNS
    """
    find_sns_region = OUTBOUND_TOPIC_ARN.split(":")
    sns_region = find_sns_region[3]
    subject = "ElasticSearch Logging Compliance Report"
    message = messagebody
    message += "\n\n"
    message += "This notification was generated by the Lambda function " + \
        context.invoked_function_arn
    send_client = boto3.client('sns', region_name=sns_region)
    try:
        send_client.publish(
            TopicArn=OUTBOUND_TOPIC_ARN,
            Message=message,
            Subject=subject
        )
    except ClientError as err:
        print(err)
        return False